// testgen_full.cpp
#include <bits/stdc++.h>
using namespace std;
using ll = long long;
static const ll MAXV = 1000000000LL;

struct Test
{
    std::vector <ll> A;
    std::vector <ll> B;
    std::string note;
};

std::vector <Test> tests;
std::vector <int> indices;

// ------------ verifier: user's function (split-based) -------------
string solve_verifier(const vector<pair<int,int>> &s) {
    int n = (int)s.size();
    if (n == 0) return string();
    vector<int> perm(n);
    iota(perm.begin(), perm.end(), 0);
    sort(perm.begin(), perm.end(), [&](int x, int y){
        if (s[x].first != s[y].first) return s[x].first > s[y].first;
        return s[x].second > s[y].second;
    });
    vector<int> prefMIN(n), suffMAX(n);
    string ans(n, 'L');
    prefMIN[0] = s[perm[0]].second;
    for (int i = 1; i < n; ++i) prefMIN[i] = min(prefMIN[i-1], s[perm[i]].second);
    suffMAX[n-1] = s[perm[n-1]].second;
    for (int i = n-2; i >= 0; --i) suffMAX[i] = max(suffMAX[i+1], s[perm[i]].second);
    for (int i = 0; i < n; ++i) {
        ans[perm[i]] = 'W';
        if (i+1 < n && s[perm[i+1]].first != s[perm[i]].first && prefMIN[i] > suffMAX[i+1]) break;
    }
    return ans;
}

// ------------ file helpers -------------
static void write_input_file(const string &fname, int N, const vector<ll>& A, const vector<ll>& B) {
    FILE *f = fopen(fname.c_str(), "wb");
    if (!f) { fprintf(stderr, "Cannot open %s for writing\n", fname.c_str()); exit(1); }
    fprintf(f, "%d\n", N);
    for (int i = 0; i < N; ++i) fprintf(f, "%lld %lld\n", A[i], B[i]);
    fclose(f);
}
static void write_solution_file(const string &fname, const string &sol) {
    FILE *f = fopen(fname.c_str(), "wb");
    if (!f) { fprintf(stderr, "Cannot open %s for writing\n", fname.c_str()); exit(1); }
    fprintf(f, "%s\n", sol.c_str());
    fclose(f);
}
static void write_perm_file(const string &fname, const vector<int>& p) {
    FILE *f = fopen(fname.c_str(), "wb");
    if (!f) { fprintf(stderr, "Cannot open %s for writing\n", fname.c_str()); exit(1); }
    for (size_t i = 0; i < p.size(); ++i) {
        if (i) fputc(' ', f);
        fprintf(f, "%d", p[i]);
    }
    fputc('\n', f);
    fclose(f);
}
static void append_to_log(const string &line) {
    FILE *f = fopen("test_log.txt", "a");
    if (!f) return;
    fprintf(f, "%s", line.c_str());
    fclose(f);
}

// ------------ utility: clamp and N per subtask -------------
int N_for_subtask(int sub) {
    if (sub == 0) return 4;
    if (sub == 1) return 200000;
    if (sub == 2) return 5;
    if (sub == 3) return 500;
    if (sub == 4) return 3000;
    if (sub == 5) return 200000;
    if (sub == 6) return 1000000;
    return 0;
}
int count_tests_for_subtask(int sub) {
    if (sub == 0) return 1;
    if (sub == 1) return 7; // as discussed earlier
    int N = N_for_subtask(sub);
    int maxk = 0; while ((1 << 2 * (maxk+1)) <= N) ++maxk;
    int main_count = maxk + 1;
    int antitest_count = 10;
    return main_count + antitest_count;
}

// ------------ permutation builder (randomized, preserves s) -------------
vector<int> build_perm_for_s_randomized(int N, int s, mt19937 &rng) {
    if (s < 0) s = 0;
    if (s >= N) s = N-1;
    vector<int> p(N, -1);

    // prefix: random permutation of 1..s
    if (s > 0) {
        p.resize(N);
        for (int i = 0; i < s; ++i) p[i] = i+1;
        shuffle(p.begin(), p.begin()+s, rng);
    } else {
        p.resize(N);
    }

    // rem: values s+1..N
    vector<int> rem;
    rem.reserve(max(0, N - s));
    for (int v = s+1; v <= N; ++v) rem.push_back(v);
    vector<int> pos_in_rem(N+1, -1);
    for (int i = 0; i < (int)rem.size(); ++i) pos_in_rem[rem[i]] = i;

    int current_max = s;
    for (int pos = s; pos < N; ++pos) {
        int forbidden = -1;
        if (current_max < pos+1 && pos+1 <= N) forbidden = pos+1;

        int r;
        if (forbidden == -1 || pos_in_rem[forbidden] == -1) {
            r = uniform_int_distribution<int>(0, (int)rem.size()-1)(rng);
        } else {
            // avoid forbidden: pick random index, if it points to forbidden swap with last (avoids bias)
            int sz = (int)rem.size();
            r = uniform_int_distribution<int>(0, sz-1)(rng);
            if (rem[r] == forbidden) {
                if (sz == 1) {
                    // cannot avoid; pick it (rare)
                    r = 0;
                } else {
                    // swap rem[r] with rem[sz-1] so we pick a different value
                    swap(rem[r], rem[sz-1]);
                    pos_in_rem[rem[r]] = r;
                    pos_in_rem[rem[sz-1]] = sz-1;
                    // pick the swapped-in at index r (now not forbidden)
                }
            }
        }
        int val = rem[r];
        p[pos] = val;
        // remove rem[r]
        int last = rem.back();
        rem[r] = last;
        pos_in_rem[last] = r;
        rem.pop_back();
        pos_in_rem[val] = -1;
        if (val > current_max) current_max = val;
    }
    return p;
}

// deterministic fallback
vector<int> build_perm_for_s_deterministic(int N, int s) {
    vector<int> p(N, -1);
    for (int i=0;i<s;++i) p[i] = i+1;
    int cur = N;
    for (int pos = s; pos <= N-2; ++pos) p[pos] = cur--;
    p[N-1] = s+1;
    return p;
}
vector<int> build_perm_no_prefix_record(int N, mt19937 &rng) {
    vector<int> p(N);
    iota(p.begin(), p.end(), 1);
    shuffle(p.begin(), p.end(), rng);
    // move 1 to last position to avoid prefix-records
    auto it = find(p.begin(), p.end(), 1);
    if (it != p.end()) rotate(it, it+1, p.end());
    return p;
}

// ------------ mapping p -> A,B,intended (randomized buckets, safe ranges) -------------
void build_arrays_from_perm_and_s(int N, const vector<int>& p, int s, mt19937 &rng,
                                  vector<ll>& A, vector<ll>& B, vector<char>& intended,
                                  bool force_block_gap = true)
{
    A.assign(N, 0);
    B.assign(N, 0);
    intended.assign(N, 'L');
    // bucket gap
    ll base_gap = max(1LL, MAXV / (N + 1LL));
    // B ranges
    const int LOW_MAX = 400000000;
    const int HIGH_MIN = 600000000;
    uniform_int_distribution<int> low_dist(0, LOW_MAX);
    uniform_int_distribution<int> high_dist(HIGH_MIN, (int)MAXV);

    for (int pos = 0; pos < N; ++pos) {
        int idx = p[pos] - 1;
        // A: make larger for smaller pos so sorting by A desc recovers permutation
        ll bucket = (ll)(N - pos) * base_gap;
        int delta = (int)uniform_int_distribution<int>(0, (int)base_gap - 1)(rng);
        A[idx] = bucket + delta;
        // B: left block (pos < s) get high, right get low; if s==0 we'll handle below
        if (s >= 1) {
            if (pos < s) B[idx] = high_dist(rng);
            else B[idx] = low_dist(rng);
            intended[idx] = (pos < s ? 'W' : 'L');
        } else {
            // s == 0: set identical B to avoid any split (all W)
            B[idx] = MAXV / 2;
            intended[idx] = 'W';
        }
    }
    // if force_block_gap==false, we keep randomness; if true ensure min(left) > max(right)
    if (force_block_gap && s >= 1) {
        // compute left min and right max and adjust only if needed (rare)
        int left_min = INT_MAX;
        int right_max = INT_MIN;
        for (int pos=0; pos<N; ++pos) {
            int idx = p[pos] - 1;
            if (pos < s) left_min = min(left_min, (int)B[idx]);
            else right_max = max(right_max, (int)B[idx]);
        }
        if (!(left_min > right_max)) {
            // force by resetting ranges deterministically, still random-like
            for (int pos=0; pos<N; ++pos) {
                int idx = p[pos] - 1;
                if (pos < s) B[idx] = uniform_int_distribution<int>(HIGH_MIN, (int)MAXV)(rng);
                else B[idx] = uniform_int_distribution<int>(0, LOW_MAX)(rng);
            }
        }
    }
}

// ------------ helper: write test (uses solve_verifier) -------------
void write_test_files_for_real(int id, const vector<ll>& A, const vector<ll>& B, const string &note) {
    int N = (int)A.size();
    std::cout << "here: " << A.size() << ' ' << B.size() << '\n'; 
    char inname[64], solname[64];
    sprintf(inname, "fight.%03d.in", id);
    sprintf(solname, "fight.%03d.sol", id);

    std::vector <ll> a(N), b(N);
    std::mt19937 rng(69420 + id + A[0] ^ B[0]);
    vector<pair<int,int>> s(N);
    for (int i=0;i<N;++i) s[i] = {(int)A[i], (int)B[i]};
    std::shuffle(s.begin(), s.end(), rng);
    for (int i=0;i<N;++i)
    {
        a[i] = s[i].first;
        b[i] = s[i].second;
    }

    write_input_file(inname, N, a, b);
    string ans = solve_verifier(s);
    write_solution_file(solname, ans);

    int w = 0;
    for (char c : ans) if (c == 'W') ++w;
    printf("Wrote %s / %s (N=%d) note=%s Winners=%d\n", inname, solname, N, note.c_str(), w);
    char logline[256];
    snprintf(logline, sizeof(logline), "fight.%03d.in Winners=%d note=%s\n", id, w, note.c_str());
    append_to_log(string(logline));
}

void write_test_files(int id, const vector<ll>& A, const vector<ll>& B, const vector<int>& perm_preview, const string &note)
{
    tests.push_back({A, B, note});
    indices.push_back(id);
}

// ------------ antitest builders (examples) -------------
void build_fragile_chain(int N, vector<ll>& A, vector<ll>& B) {
    A.assign(N,0); B.assign(N,0);
    for (int i=0;i<N;++i) { A[i] = 2LL*(i+1); B[i] = 2LL*(N-(i+1)) + 1; }
    for (int i=0;i+2<N;++i) B[i+2] = min(B[i+2], B[i+1]-1);
}
void build_decompression_pitfall(int N, vector<ll>& A, vector<ll>& B) {
    A.assign(N,0); B.assign(N,0);
    for (int i=0;i<N;++i) { A[i]=i+1; B[i]=i+1; }
    int block = max(1, N/100);
    for (int i=0;i<N;i+=block) {
        ll val = (ll)(i/block + 1) * 1000000;
        for (int j=i;j<min(N,i+block);++j) { A[j]=val; B[j]=val; }
    }
}
void build_alternating_SCCs(int N, vector<ll>& A, vector<ll>& B) {
    A.assign(N,0); B.assign(N,0);
    for (int k=0; 2*k+1 < N; ++k) {
        int i = 2*k, j = 2*k+1;
        A[i] = MAXV - k; B[i] = k;
        A[j] = k; B[j] = MAXV - k;
    }
    if (N % 2 == 1) { int i = N-1; A[i] = MAXV/2; B[i] = MAXV/2; }
}

// ------------ main generator -------------
int main(int argc, char** argv) {
    if (argc < 2) { fprintf(stderr, "Usage: %s <subtask_id> [seed]\n", argv[0]); return 1; }
    int sub = atoi(argv[1]);
    unsigned seed;
    if (argc >= 3) seed = (unsigned)atoi(argv[2]);
    else { random_device rd; seed = rd(); }
    mt19937 rng(seed);
    fprintf(stderr, "Seed = %u\n", seed);

    int maxN = N_for_subtask(sub);
    if (maxN == 0) { fprintf(stderr, "Bad subtask id\n"); return 1; }

    // compute starting global index
    int global_idx = 1;
    for (int t = 0; t < sub; ++t) global_idx += count_tests_for_subtask(t);
    int start_idx = global_idx;

    // Subtask 0: sample
    if (sub == 0) {
        vector<ll> A={4,1,3,2}, B={4,3,7,6};
        write_test_files_for_real(global_idx, A, B, "sample");
        printf("Subtask 0 produced fight.%03d\n", global_idx);
        return 0;
    }

    // Subtask 1: A == B tests
    if (sub == 1) {
        int N = maxN;
        vector<ll> A,B;
        // winners = 1
        build_arrays_from_perm_and_s(N, vector<int>(N,1), 1, rng, A, B, *(new vector<char>()), true); // quick: will be overwritten
        // But for A==B we should set exactly top_cnt equal to MAXV
        auto build_AeqB = [&](int top){
            A.assign(N,0); B.assign(N,0);
            for (int i=0;i<top;++i) A[i]=B[i]=MAXV;
            for (int i=top;i<N;++i) A[i]=B[i]=i+1;
        };
        build_AeqB(1); write_test_files(global_idx++, A, B, {}, "A==B winners=1");
        build_AeqB(N); write_test_files(global_idx++, A, B, {}, "A==B winners=N");
        build_AeqB(2); write_test_files(global_idx++, A, B, {}, "A==B winners=2");
        int top = max(1, (int)floor(sqrt((double)N))); build_AeqB(top); write_test_files(global_idx++, A, B, {}, "A==B winners=sqrtN");
        top = max(1, (int)floor((double)N / max(1.0, log((double)N)))); build_AeqB(top); write_test_files(global_idx++, A, B, {}, "A==B winners=N_logN");
        uniform_int_distribution<int> rdist(1, N);
        build_AeqB(rdist(rng)); write_test_files(global_idx++, A, B, {}, "A==B winners=random");
        build_decompression_pitfall(N, A, B);
        write_test_files(global_idx++, A, B, {}, "A==B heavy-eq (antitest)");
        std::cout << "HEEEEEEEEEEEEEREEEEEEEEEEEEEEEE: " << tests.size() << ' ' << indices.size() << '\n';
        for (int i = 0 ; i < tests.size() ; ++i)
        {
            write_test_files_for_real(indices[i], tests[i].A, tests[i].B, tests[i].note);
        }

        printf("Subtask 1: produced fight.%03d .. fight.%03d\n", start_idx, global_idx-1);
        return 0;
    }

    // Subtasks 2..6
    if (sub >= 2 && sub <= 6) {
        int N = maxN;
        int maxk = 0; while ((1 << 2 * (maxk+1)) <= N) ++maxk;

        // main dyadic tests: choose s = 2^k (low), clamp to [1, N-1], randomized perms
        for (int k = 0; k <= maxk; ++k) {
            int low = (1 << 2 * k);
            int s = rng() % low + low;
            if (s < 1) s = 1;
            if (s > N-1) s = max(1, N-1);

            vector<int> p = build_perm_for_s_randomized(N, s, rng);
            // verify s_found; fallback if needed
            int curmax=0, s_found=0;
            for (int i=0;i<N-1;++i){ curmax = max(curmax, p[i]); if (curmax == i+1) s_found = i+1; }
            if (s_found != s) {
                p = build_perm_for_s_deterministic(N, s);
                curmax=0; s_found=0;
                for (int i=0;i<N-1;++i){ curmax = max(curmax, p[i]); if (curmax == i+1) s_found = i+1; }
                if (s_found != s) {
                    fprintf(stderr, "Failed to build perm with s=%d N=%d\n", s, N);
                    exit(1);
                }
            }

            // build arrays
            vector<ll> A,B; vector<char> intended;
            build_arrays_from_perm_and_s(N, p, s, rng, A, B, intended, true);

            // write
            write_test_files(global_idx, A, B, p, "main dyadic");
            ++global_idx;
        }

        // Antitests (8 examples)
        // 1 fragile chain
        {
            vector<ll> A,B; build_fragile_chain(N, A, B);
            write_test_files(global_idx++, A, B, {}, "fragile-eat-chain");
        }
        // 2 tie-critical blocks
        {
            vector<ll> A,B; build_decompression_pitfall(N, A, B);
            write_test_files(global_idx++, A, B, {}, "decomp-eq");
        }
        // 3 alternating SCCs
        {
            vector<ll> A,B; build_alternating_SCCs(N, A, B);
            write_test_files(global_idx++, A, B, {}, "alt-SCC");
        }
        // 4 single dominated
        {
            vector<ll> A(N), B(N);
            for (int i=0;i<N;++i) { A[i] = MAXV - i; B[i] = MAXV - i; }
            A[N/2] = 0; B[N/2] = 0;
            write_test_files(global_idx++, A, B, {}, "single-dominated");
        }
        // 5 perm availability trap
        {
            int s = max(1, min(N-2, N/3));
            vector<int> p = build_perm_for_s_randomized(N, s, rng);
            vector<ll> A,B; vector<char> intended;
            build_arrays_from_perm_and_s(N, p, s, rng, A, B, intended, true);
            write_test_files(global_idx++, A, B, p, "perm-availability-trap");
        }
        // 6 narrow-large intervals
        {
            vector<ll> A(N), B(N);
            int i=0, toggle=0;
            while (i<N) {
                if (toggle==0) {
                    int len = min(N-i, 3);
                    for (int k=0;k<len;++k) { A[i]=i+100; B[i]=10000000 - (i+k); ++i; }
                } else {
                    int len = min(N-i, max(1, N/10));
                    for (int k=0;k<len;++k) { A[i]=i+200; B[i]=5000000 - (i+k); ++i; }
                }
                toggle ^= 1;
            }
            write_test_files(global_idx++, A, B, {}, "narrow-large");
        }
        // 7 random-direct (random permutation, direct arrays)
        {
            vector<int> p(N); iota(p.begin(), p.end(), 1); shuffle(p.begin(), p.end(), rng);
            vector<ll> A,B; vector<char> intended;
            build_arrays_from_perm_and_s(N, p, 0, rng, A, B, intended, false); // s=0 -> all W
            write_test_files(global_idx++, A, B, p, "random-direct-s0");
        }
        // 8 random small test (another random perm)
        {
            vector<int> p(N); iota(p.begin(), p.end(), 1); shuffle(p.begin(), p.end(), rng);
            vector<ll> A,B; vector<char> intended;
            int s = max(1, min(N-1, N / 10));
            build_arrays_from_perm_and_s(N, p, s, rng, A, B, intended, true);
            write_test_files(global_idx++, A, B, p, "random-small-s");
        }

        // 9
        {
            int n = N;
            std::vector <int> p;
            vector<ll> A(n),B(n);

            for (int i = 0 ; i < B.size() ; ++i)
            {
                B[i] = i;
            }

            for (int i = 0 ; i < A.size() ; ++i)
            {
                A[i] = rng() % ((int)sqrt(n));
            }
            
            std::sort(A.begin(), A.end());
            write_test_files(global_idx++, A, B, p, "myantitest-A");
        }

         // 10
        {
            int n = N;
            std::vector <int> p;
            vector<ll> A(n),B(n);

            for (int i = 0 ; i < B.size() ; ++i)
            {
                B[i] = i;
            }

            for (int i = 0 ; i < A.size() ; ++i)
            {
                A[i] = rng() % ((int)sqrt(n));
            }
            
            std::sort(A.begin(), A.end());
            std::swap(A, B);
            write_test_files(global_idx++, A, B, p, "myantitest-B");
        }

        if (sub == 4)
        {
            int n = N;
            std::vector <int> p;
            vector<ll> A(n),B(n);

            for (int i = 0 ; i < B.size() ; ++i)
            {
                B[i] = rng() % ((int)1e9);
            }

            for (int i = 0 ; i < A.size() ; ++i)
            {
                A[i] = rng() % ((int)1e9);
            }
            
            std::sort(A.begin(), A.end());
            std::swap(A, B);
            write_test_files(global_idx++, A, B, p, "myantitest-B");
        }

        std::shuffle(tests.begin(), tests.end(), rng);
        std::cout << "HEEEEEEEEEEEEEREEEEEEEEEEEEEEEE: " << tests.size() << ' ' << indices.size() << '\n';
        for (int i = 0 ; i < tests.size() ; ++i)
        {
            write_test_files_for_real(indices[i], tests[i].A, tests[i].B, tests[i].note);
        }
        printf("Subtask %d: produced fight.%03d .. fight.%03d (N=%d)\n", sub, start_idx, global_idx-1, N);
        return 0;
    }

    fprintf(stderr, "Unhandled subtask id\n");
    return 1;
}
