// interval_mst_prefmin.cpp
// Merge-sort tree storing (sorted B-values) and prefix-min position per node.
// For each x we find leftmost y < x with B[y] <= B[x] (or same-A earliest).
// Complexity: O(N log^2 N) worst-case.
//
// Compile: g++ -O2 -std=c++17 interval_mst_prefmin.cpp -o interval_mst_prefmin

#include <bits/stdc++.h>
using namespace std;
using vi = vector<int>;
using pii = pair<int,int>;
const int INF = 1e9;

struct MergeNode {
    vector<int> vals;      // sorted B-values in this node
    vector<int> prefMin;   // prefMin[i] = minimum position among entries vals[0..i]
};

struct MergeSortTree {
    int n;
    vector<MergeNode> tree;
    MergeSortTree(int n_=0){ init(n_); }
    void init(int n_){
        n = n_;
        tree.assign(4*n + 5, MergeNode());
    }

    // build from array of pairs (B, pos) where pos is index in sorted-by-A order
    void build(const vector<pair<int,int>>& arr){ // arr.size() == n
        build_rec(1, 0, n-1, arr);
    }

    void build_rec(int node, int l, int r, const vector<pair<int,int>>& arr){
        if(l == r){
            tree[node].vals = { arr[l].first };       // B value
            tree[node].prefMin = { arr[l].second };   // position
            return;
        }
        int mid = (l + r) >> 1;
        build_rec(node<<1, l, mid, arr);
        build_rec(node<<1|1, mid+1, r, arr);

        auto &L = tree[node<<1].vals;
        auto &R = tree[node<<1|1].vals;
        auto &Lp = tree[node<<1].prefMin;
        auto &Rp = tree[node<<1|1].prefMin;

        int ls = (int)L.size(), rs = (int)R.size();
        tree[node].vals.resize(ls + rs);
        // also build prefMin aligned with merged vals
        tree[node].prefMin.resize(ls + rs);

        // merge vals, and compute prefix-min of positions
        int i=0,j=0,k=0;
        int curMin = INF;
        while(i<ls || j<rs){
            if(j==rs || (i<ls && L[i] <= R[j])){
                tree[node].vals[k] = L[i];
                curMin = std::min(curMin, Lp[i]);
                tree[node].prefMin[k] = curMin;
                ++i; ++k;
            } else {
                tree[node].vals[k] = R[j];
                curMin = std::min(curMin, Rp[j]);
                tree[node].prefMin[k] = curMin;
                ++j; ++k;
            }
        }
    }

    // Query range [ql, qr] (0-based positions) for minimal position among entries with value <= val.
    // Returns INF if none.
    int query_min_pos_leq(int ql, int qr, int val){ 
        if(ql>qr) return INF;
        return query_rec(1, 0, n-1, ql, qr, val);
    }

    int query_rec(int node, int l, int r, int ql, int qr, int val){
        if(qr < l || r < ql) return INF;
        if(ql <= l && r <= qr){
            // in this node find rightmost index p such that vals[p] <= val
            auto &V = tree[node].vals;
            if(V.empty() || V[0] > val) return INF;
            int p = int(upper_bound(V.begin(), V.end(), val) - V.begin()) - 1;
            return tree[node].prefMin[p];
        }
        int mid = (l + r) >> 1;
        int leftRes = query_rec(node<<1, l, mid, ql, qr, val);
        int rightRes = query_rec(node<<1|1, mid+1, r, ql, qr, val);
        return min(leftRes, rightRes);
    }
};

int main(){
    ios::sync_with_stdio(false);
    cin.tie(nullptr);
    int N;
    if(!(cin>>N)) return 0;
    vector<pair<int,int>> orig(N);
    for(int i=0;i<N;++i) cin>>orig[i].first>>orig[i].second;

    // sort by A desc, tie B desc
    vector<int> idx(N);
    iota(idx.begin(), idx.end(), 0);
    sort(idx.begin(), idx.end(), [&](int x,int y){
        if(orig[x].first != orig[y].first) return orig[x].first > orig[y].first;
        return orig[x].second > orig[y].second;
    });

    // build array arr of (B, pos) where pos = sorted order index
    vector<pair<int,int>> arr(N);
    for(int i=0;i<N;++i) arr[i] = { orig[idx[i]].second, i };

    MergeSortTree mst(N);
    if(N>0) mst.build(arr);

    // map A -> earliest position in sorted order
    unordered_map<int,int> firstPosA;
    firstPosA.reserve(N*2);
    for(int i=0;i<N;++i){
        int a = orig[idx[i]].first;
        if(firstPosA.find(a) == firstPosA.end()) firstPosA[a] = i;
    }

    // For each x find earliest y < x with B[y] <= B[x] via MST query on [0..x-1]
    vector<int> maxEndForStart(N, -1);
    for(int x=0;x<N;++x){
        int bestY = INF;
        if(x>0){
            int q = mst.query_min_pos_leq(0, x-1, arr[x].first);
            if(q != INF) bestY = min(bestY, q);
        }
        // same A earliest
        int aval = orig[idx[x]].first;
        auto it = firstPosA.find(aval);
        if(it != firstPosA.end()){
            int p = it->second;
            if(p < x) bestY = min(bestY, p);
        }
        if(bestY != INF){
            if(maxEndForStart[bestY] < x) maxEndForStart[bestY] = x;
        }
    }

    // sweep to find smallest k such that max_end_so_far <= k
    int max_end_so_far = -1;
    int found_k = -1;
    for(int k=0;k<N;++k){
        if(maxEndForStart[k] > max_end_so_far) max_end_so_far = maxEndForStart[k];
        if(max_end_so_far <= k){
            found_k = k;
            break;
        }
    }

    string ans(N, 'L');
    if(found_k != -1){
        for(int i=0;i<=found_k;++i) ans[idx[i]] = 'W';
    }
    cout << ans << '\n';
    return 0;
}
