#include "common.h"
#include <bits/stdc++.h>

template <int64_t COUNT>
constexpr auto generateRatios(const double start = 0.0, const double end = 1.0) {
    std::array<double, COUNT + 1> ret;
    for (int64_t i = 0; i <= COUNT; i++) {
        ret[i] = (end - start) * (double)(i + 1) / (double)COUNT + start;
    }
    return ret;
};

constexpr auto generateRandom1N(const int64_t up) { return rnd.next(std::max(static_cast<int64_t>(1), up - up / static_cast<int64_t>(10)), up); }

struct SubtaskConstraints : BaseConstraints {
    int64_t N;
    int64_t M;
    int64_t R;
    int64_t C;

    void validate(std::istream& in) const {
        int64_t n, m, r, c;
        in >> n >> m >> r >> c;
        assert(1 <= n && n <= N);
        assert(1 <= m && m <= M);
        assert(1 <= r && r <= R);
        assert(1 <= c && c <= C);
        std::string s;
        for (int64_t i = 0; i < n; i++) {
            in >> s;
            assert(static_cast<int64_t>(s.size()) == m);
            for (char ch : s)
                assert(ch == '0' || ch == '1');
        }
    }
};

bool finalize(std::ostream& out, int64_t n, int64_t m, int64_t r, int64_t c, const std::vector<std::string>& A) {
    out << n << " " << m << " " << r << " " << c << "\n";
    for (const auto& row : A) {
        out << row << "\n";
    }
    return true;
}

bool generateRandom(std::ostream& out, const std::optional<SubtaskConstraints>& cst, const double p = 0.5) {
    const auto cstv = cst.value();
    int64_t n = generateRandom1N(cstv.N);
    int64_t m = generateRandom1N(cstv.M);
    int64_t r = generateRandom1N(cstv.R);
    int64_t c = generateRandom1N(cstv.C);

    std::vector<std::string> A(n, std::string(m, '0'));
    for (int64_t i = 0; i < n; i++) {
        for (int64_t j = 0; j < m; j++) {
            A[i][j] = (rnd.next(0.0, 1.0) < p ? '1' : '0');
        }
    }

    return finalize(out, n, m, r, c, A);
}

bool generateChecker(std::ostream& out, const std::optional<SubtaskConstraints>& cst) {
    const auto cstv = cst.value();
    int64_t n = generateRandom1N(cstv.N);
    int64_t m = generateRandom1N(cstv.M);
    int64_t r = generateRandom1N(cstv.R);
    int64_t c = generateRandom1N(cstv.C);

    std::vector<std::string> A(n, std::string(m, '0'));
    for (int64_t i = 0; i < n; i++) {
        for (int64_t j = 0; j < m; j++) {
            if ((i + j) & 1) {
                A[i][j] = '1';
            }
        }
    }

    return finalize(out, n, m, r, c, A);
}

bool generateStripes(std::ostream& out, const std::optional<SubtaskConstraints>& cst, const bool flip, const double p) {
    const auto cstv = cst.value();
    int64_t n = generateRandom1N(cstv.N);
    int64_t m = generateRandom1N(cstv.M);
    int64_t r = generateRandom1N(cstv.R);
    int64_t c = generateRandom1N(cstv.C);
    std::vector<std::string> A(n, std::string(m, '0'));

    if (static_cast<int64_t>(rnd.next(0.0, 1.0))) {
        for (int64_t i = 0; i < n; i++) {
            const auto on = (rnd.next(0.0, 1.0) < p ? '1' : '0');
            for (int64_t j = 0; j < m; j++) {
                if (on) {
                    A[i][j] = '1';
                }
            }
        }
    } else {
        for (int64_t j = 0; j < m; j++) {
            const auto on = (rnd.next(0.0, 1.0) < p ? '1' : '0');
            for (int64_t i = 0; i < n; i++) {
                if (on) {
                    A[i][j] = '1';
                }
            }
        }
    }

    return finalize(out, n, m, r, c, A);
}

bool generateStaircase(std::ostream& out, const std::optional<SubtaskConstraints>& cst, const bool invert_horizontal, const bool invert_vertical,
                       const bool do_strike) {
    const auto cstv = cst.value();
    int64_t n = generateRandom1N(cstv.N);
    int64_t m = generateRandom1N(cstv.M);
    int64_t r = generateRandom1N(cstv.R);
    int64_t c = generateRandom1N(cstv.C);

    std::vector<std::string> A(n, std::string(m, '0'));

    int64_t steps = std::min(n, m);

    for (int64_t i = 0; i < n; i++) {
        for (int64_t j = 0; j < m; j++) {
            if (i <= j) {
                A[i][j] = '1';
            }
        }
    }

    if (invert_horizontal) {
        for (int64_t i = 0; i < n; i++) {
            for (int64_t j = i; j < m; j++) {
                A[i][j] = A[i][m - j - 1];
            }
        }
    }

    if (invert_vertical) {
        for (int64_t i = 0; i < n; i++) {
            for (int64_t j = i; j < m; j++) {
                A[i][j] = A[n - i - 1][j];
            }
        }
    }

    for (int64_t i = 0; i < n; i++) {
        for (int64_t j = 0; j < m; j++) {
            if (i == 0 || j == 0 || i == n - 1 || j == m - 1) {
                A[i][j] = '0';
            }
        }
    }

    if (do_strike) {
        int64_t row = rnd.next(static_cast<int64_t>(0), n - 1);
        A[row] = std::string(m, '0');
    }

    return finalize(out, n, m, r, c, A);
}

bool generateLine(std::ostream& out, const std::optional<SubtaskConstraints>& cst, const bool horizontal, const int64_t num_blocked, const int64_t num_lines) {
    const auto cstv = cst.value();
    int64_t n = cstv.N;
    int64_t m = cstv.M;
    int64_t r = cstv.R;
    int64_t c = cstv.C;

    if (horizontal) {
        n = num_lines;
    } else {
        m = num_lines;
    }

    std::vector<std::pair<int64_t, int64_t>> all_cells;

    for (int64_t i = 0; i < n; i++) {
        for (int64_t j = 0; j < m; j++) {
            all_cells.emplace_back(i, j);
        }
    }
    shuffle(all_cells.begin(), all_cells.end());

    std::vector<std::string> A(n, std::string(m, '1'));
    for (int64_t i = 0; i < num_blocked && i < all_cells.size(); i++) {
        A[all_cells[i].first][all_cells[i].second] = '0';
    }

    return finalize(out, n, m, r, c, A);
}

bool generateDegenerate(std::ostream& out, const std::optional<SubtaskConstraints>& cst, const bool horizontal, const bool vertical) {
    const auto cstv = cst.value();
    int64_t n = generateRandom1N(cstv.N);
    int64_t m = generateRandom1N(cstv.M);
    int64_t r = generateRandom1N(cstv.R);
    int64_t c = generateRandom1N(cstv.C);

    if (!horizontal) {
        r = 1;
    }

    if (!vertical) {
        c = 1;
    }

    std::vector<std::string> A(n, std::string(m, '0'));

    A[0][0] = '1';
    A[0][m - 1] = '1';
    A[n - 1][0] = '1';
    A[n - 1][m - 1] = '1';

    return finalize(out, n, m, r, c, A);
}

signed main(int argc, char** argv) {
    // clang-format off

    registerGen(argc, argv, 1);

	#define withAllTestGensSingleRow(TIMES) \
		withTestGens( \
			TIMES, nameFun(generateRandom), \
            std::vector<double>({-0.1, 0.5, 1.1}) \
		). \
		withTestGens( \
			nameFun(generateStripes), \
            std::vector<bool>({false, true}), std::vector<double>({0.3, 0.6}) \
		)

    #define withAllTestGens(TIMES) \
        withAllTestGensSingleRow(TIMES) \
        .withTestGens( \
            nameFun(generateStaircase), \
            std::vector({false, true}), std::vector<bool>({false, true}), std::vector<bool>({false, true}) \
        ) \
        .withTestGens( \
            nameFun(generateLine), \
            std::vector<bool>({false, true}), std::vector<int64_t>({10, 30, 1000}), std::vector<int64_t>({2}) \
        )

    auto subtask0 = Subtask<SubtaskConstraints>("Sample", 0)
        .withConstraints({.N=10, .M=10, .R=10, .C=10})
        .withTestGens(nameFun(fileTestGen<SubtaskConstraints>),
            std::vector<std::string>{"testgen/extras/sample_1.in"});

    auto subtask1 = Subtask<SubtaskConstraints>("N=1, R=C=1", 10)
        .withConstraints({.N=1, .M=2000, .R=1, .C=1})
        .withTestGens(nameFun(generateLine), std::vector<bool>({true}), std::vector<int64_t>({10, 30, 1000}), std::vector<int64_t>({1}))
        .withAllTestGensSingleRow(1);

    auto subtask2 = Subtask<SubtaskConstraints>("N=1, R=1", 5)
        .withConstraints({.N=1, .M=2000, .R=1, .C=1000000})
        .dependsOn(subtask1)
        .withTestGens(nameFun(generateLine), std::vector<bool>({true}), std::vector<int64_t>({10, 30, 1000}), std::vector<int64_t>({1}))
        .withAllTestGensSingleRow(1);

    auto subtask3 = Subtask<SubtaskConstraints>("N=1, C=1", 5)
        .withConstraints({.N=1, .M=2000, .R=1000000, .C=1})
        .dependsOn(subtask1)
        .withTestGens(nameFun(generateLine), std::vector<bool>({true}), std::vector<int64_t>({10, 30, 1000}), std::vector<int64_t>({1}))
        .withAllTestGensSingleRow(1);

    auto subtask4 = Subtask<SubtaskConstraints>("N, M<=20, R=C=1", 10)
        .withConstraints({.N=20, .M=20, .R=1, .C=1})
        .withAllTestGens(1);

    auto subtask5 = Subtask<SubtaskConstraints>("N, M<=60, R=C=1", 15)
        .withConstraints({.N=60, .M=60, .R=1, .C=1})
        .dependsOn(subtask4)
        .withAllTestGens(1);

    auto subtask6 = Subtask<SubtaskConstraints>("N, M<=300, R=C=1", 15)
        .withConstraints({.N=300, .M=300, .R=1, .C=1})
        .dependsOn(subtask5)
        .withAllTestGens(1);

    auto subtask7 = Subtask<SubtaskConstraints>("N, M<=2000, R=C=1", 10)
        .withConstraints({.N=2000, .M=2000, .R=1, .C=1})
        .dependsOn(subtask6)
        .withAllTestGens(1);

    auto subtask8 = Subtask<SubtaskConstraints>("Full", 30)
        .withConstraints({.N=2000, .M=2000, .R=1000000, .C=1000000})
        .dependsOn(subtask7)
        .withAllTestGens(1);

    std::vector<Subtask<SubtaskConstraints>> subtasks = {
        subtask0,
        subtask1,
        subtask2,
        subtask3,
        subtask4,
        subtask5,
        subtask6,
        subtask7,
        subtask8
    };

    const auto author = Solution("author");
    const auto slow = Solution("subtask5");

	if (false) {
		const std::vector<Subtask<SubtaskConstraints>> stressSubtasks = {
			subtask5,
		};
		stressTest(stressSubtasks, author, slow);
	} else {
        generateSubtasks("matrix", subtasks, Solution("author"));
        generateGradeProperties(subtasks, "256", "0.6");
	}
}
