// gen_elections_reduced_repr_shuffled_fixed_s2.cpp
// Variant of gen_elections_reduced_repr_shuffled.cpp with one change:
// - For subtask 2: restore the "all-equal" representative test,
//   and replace the previous "permutation 1..N" rep with an alternating periodic pattern.
// - Keeps deterministic per-file shuffle and other behavior.
//
// Compile: g++ -std=c++17 -O2 gen_elections_reduced_repr_shuffled_fixed_s2.cpp -o gen_elections_reduced_repr_shuffled_fixed_s2
// Run: ./gen_elections_reduced_repr_shuffled_fixed_s2

#include <bits/stdc++.h>
using namespace std;
using ull = unsigned long long;
using ll = long long;

const ull BASE_SEED = 123456789ULL;
const int MOD = 1'000'000'007;

// Subtask maximum N
const int MAXN_SUB[8] = {0, 8, 20, 100000, 100000, 1000, 100000, 2000000};
// Tests per subtask: sub1 = 2 (samples), others = 6 (5 fixed + 1 random)
const int TESTS_PER_SUB[8] = {0,2,6,6,6,6,6,6};

// rng helpers
int randint_mt(std::mt19937_64 &rng, int a, int b){
    uniform_int_distribution<int> d(a,b);
    return d(rng);
}
double next01(std::mt19937_64 &rng){
    uniform_real_distribution<double> d(nextafter(0.0,1.0), 1.0);
    return d(rng);
}
void shuffle_with_rng(vector<int>& v, std::mt19937_64 &rng){ shuffle(v.begin(), v.end(), rng); }

void write_test_file_shuffled(int idx, int subtask, vector<int> P){
    // per-file deterministic shuffle (so files are reproducibly different)
    std::mt19937_64 rng(BASE_SEED + (ull)idx * 1315423911ULL + (ull)subtask * 10007ULL);
    if(P.size() > 1) shuffle_with_rng(P, rng);

    string fn = "elections." + to_string(idx) + ".in";
    FILE *f = fopen(fn.c_str(),"w");
    if(!f){ cerr<<"Cannot open "<<fn<<"\n"; exit(1); }
    fprintf(f, "%d\n", (int)P.size());
    for(size_t i=0;i<P.size();++i){
        if(i) fprintf(f, " ");
        fprintf(f, "%d", P[i]);
    }
    fprintf(f, "\n");
    fclose(f);
}

// --- small builders used throughout ---
vector<int> gen_all_distinct(int N){ vector<int> P(N); for(int i=0;i<N;++i) P[i]=i+1; return P; }
vector<int> gen_all_equal(int N,int x){ if(x<1||x> N) x=1; return vector<int>(N,x); }
vector<int> gen_random_values(int N,int K,std::mt19937_64 &rng){ if(K<1) K=1; if(K>N) K=N; vector<int>P(N); for(int i=0;i<N;++i) P[i]=randint_mt(rng,1,K); return P; }

// build P from counts and ids, layout: 0=block,1=interleave,2=shuffle
vector<int> build_from_counts(const vector<int>& cnt, const vector<int>& ids, int layout, std::mt19937_64 &rng){
    int x = (int)cnt.size(); int N=0; for(int v:cnt) N+=v;
    vector<int> A; A.reserve(N);
    if(layout==0){
        for(int i=0;i<x;++i) for(int t=0;t<cnt[i];++t) A.push_back(ids[i]);
    } else if(layout==1){
        vector<int> cur(cnt.begin(), cnt.end()); int rem=N;
        while(rem>0){
            for(int i=0;i<x && rem>0;++i){
                if(cur[i]>0){ A.push_back(ids[i]); cur[i]--; rem--; }
            }
        }
    } else {
        for(int i=0;i<x;++i) for(int t=0;t<cnt[i];++t) A.push_back(ids[i]);
        shuffle_with_rng(A, rng);
    }
    return A;
}

// gen weighted counts (exp random allocation), used by weighted tests
vector<int> gen_weighted_counts(int N,int x,bool ensure_nonzero,std::mt19937_64 &rng){
    if(x<=0) x=1; if(x> N) x=N;
    vector<double> w(x), frac(x);
    double S=0;
    for(int i=0;i<x;++i){ double u = next01(rng); w[i] = -log(u); S += w[i]; }
    vector<int> cnt(x,0); int total=0;
    for(int i=0;i<x;++i){
        double real = (w[i]/S) * (double)N;
        cnt[i] = (int)floor(real);
        frac[i] = real - cnt[i];
        total += cnt[i];
    }
    int rem = N - total;
    vector<int> idx(x); iota(idx.begin(), idx.end(), 0);
    sort(idx.begin(), idx.end(), [&](int a,int b){ return frac[a] > frac[b]; });
    for(int k=0;k<rem;++k) cnt[idx[k % x]]++;
    if(ensure_nonzero && x <= N){
        vector<int> zeros;
        for(int i=0;i<x;++i) if(cnt[i]==0) zeros.push_back(i);
        if(!zeros.empty()){
            vector<int> donors(x); iota(donors.begin(), donors.end(), 0);
            sort(donors.begin(), donors.end(), [&](int a,int b){ if(cnt[a]!=cnt[b]) return cnt[a]>cnt[b]; return a<b; });
            int di=0;
            for(int z:zeros){
                while(di < (int)donors.size() && donors[di]==z) di++;
                while(di < (int)donors.size() && cnt[donors[di]] <= 1) di++;
                if(di >= (int)donors.size()) break;
                cnt[donors[di]]--; cnt[z]++; 
            }
            for(int z: zeros) if(cnt[z]==0){
                int best = max_element(cnt.begin(), cnt.end()) - cnt.begin();
                if(cnt[best] > 0){ cnt[best]--; cnt[z]++; }
            }
        }
    }
    int s = accumulate(cnt.begin(), cnt.end(), 0);
    if(s != N) cnt[0] += (N - s);
    return cnt;
}
vector<int> make_weighted_x_test(int N,int x,int layout,std::mt19937_64 &rng){
    x = max(1, min(x, N));
    vector<int> cnt = gen_weighted_counts(N, x, true, rng);
    vector<int> ids((int)cnt.size());
    for(int i=0;i<(int)ids.size();++i) ids[i]=i+1;
    return build_from_counts(cnt, ids, layout, rng);
}

// each value occurs <= 2 test
vector<int> make_each_at_most_2_test(int N,int unique_u,int layout,std::mt19937_64 &rng){
    unique_u = max(1, min(unique_u, N));
    vector<int> cnt(unique_u, 1);
    int rem = N - unique_u;
    vector<int> idx(unique_u); iota(idx.begin(), idx.end(), 0);
    shuffle_with_rng(idx, rng);
    for(int i=0;i<rem;++i) cnt[idx[i % unique_u]]++;
    vector<int> ids(unique_u); iota(ids.begin(), ids.end(), 1);
    return build_from_counts(cnt, ids, layout, rng);
}

// build_for_x2: x2 ids that occur exactly twice, rest singletons (used in subtask4)
vector<int> build_for_x2(int N,int x2,int layout,std::mt19937_64 &rng){
    if(x2 < 0) x2 = 0;
    int twos = min(x2, N/2);
    int ones = N - 2*twos;
    int u = twos + ones; if(u<=0) u=1;
    vector<int> cnt(u,0);
    for(int i=0;i<twos;++i) cnt[i]=2;
    for(int i=twos;i<u;++i) cnt[i]=1;
    vector<int> ids(u); iota(ids.begin(), ids.end(), 1);
    shuffle_with_rng(ids, rng);
    return build_from_counts(cnt, ids, layout, rng);
}

// helper pow5 list
vector<long long> pow5_list(long long max_x2){
    vector<long long> p; long long cur = 1;
    while(cur <= max_x2){ p.push_back(cur); if(cur > max_x2 / 5) break; cur *= 5; }
    return p;
}

// MAIN generation
int main(){
    ios::sync_with_stdio(false);
    cin.tie(nullptr);

    vector<pair<int, vector<int>>> all;
    auto add = [&](int sub, const vector<int>& P){ all.emplace_back(sub, P); };

    int global_index = 0;

    for(int s=1; s<=7; ++s){
        int maxN = MAXN_SUB[s];
        int total = TESTS_PER_SUB[s];

        if(s == 1){
            add(s, vector<int>{2,1,2});
            add(s, vector<int>{1,3,4,3,3,2,3,1});
            continue;
        }

        vector<vector<int>> reps; reps.reserve(5);
        std::mt19937_64 pickrng(BASE_SEED + (ull)s * 7919ULL);

        if(s == 2){
            // representative tests for N<=20
            reps.push_back(gen_all_distinct(maxN));
            // RESTORED: second test is all-equal (as requested)
            reps.push_back(gen_all_equal(maxN, 1));
            // small alphabet weighted (K ~ sqrt(N))
            int K = max(2, (int)floor(sqrt((double)maxN)));
            reps.push_back(make_weighted_x_test(maxN, K, 2, pickrng));
            // weighted with K ~ N/3
            reps.push_back(make_weighted_x_test(maxN, max(2, maxN/3), 1, pickrng));
            // REPLACED permutation -> alternating periodic pattern of small alphabet
            {
                int k = max(2, (int)floor(sqrt((double)maxN))); // alphabet size
                vector<int> P; P.reserve(maxN);
                for(int i=0;i<maxN;++i) P.push_back((i % k) + 1);
                // optionally apply small deterministic rotation for variety
                shuffle_with_rng(P, pickrng); // keep some shuffle so pattern isn't trivially ordered
                reps.push_back(P);
            }
        } else if(s == 3){
            reps.push_back(gen_all_equal(maxN, 1));
            reps.push_back(gen_all_equal(maxN, 2));
            { vector<int>P(maxN); for(int i=0;i<maxN;++i) P[i] = (i < maxN/2 ? 1 : 2); shuffle_with_rng(P, pickrng); reps.push_back(P); }
            { vector<int>P(maxN); for(int i=0;i<maxN;++i) P[i] = (i < maxN/2 + 1 ? 1 : 2); shuffle_with_rng(P, pickrng); reps.push_back(P); }
            { vector<long long> p5 = pow5_list(maxN); int bidx = (int)p5.size()/2; long long lo = p5.empty() ? 1 : p5[bidx]; long long hi = p5.empty() ? maxN : min((long long)maxN, lo*5 - 1); long long pick = (lo + hi)/2; int cnt1 = (int)min<long long>(maxN, pick); vector<int>P(maxN); for(int i=0;i<maxN;++i) P[i] = (i < cnt1 ? 1 : 2); shuffle_with_rng(P, pickrng); reps.push_back(P); }
        } else if(s == 4){
            reps.push_back(gen_all_distinct(maxN));
            reps.push_back(build_for_x2(maxN, maxN/2, 2, pickrng));
            { int max_x2 = maxN/2; vector<long long> p5 = pow5_list(max_x2); int idx = max(0, (int)p5.size()/2); long long lo = p5.empty() ? 1 : p5[idx]; long long hi = p5.empty() ? max_x2 : min((long long)max_x2, lo*5 - 1); long long pick = (lo + hi) / 2; int x2 = (int)min<long long>(max_x2, pick); reps.push_back(build_for_x2(maxN, x2, 1, pickrng)); }
            reps.push_back(make_each_at_most_2_test(maxN, max(1, maxN/2 + 10), 0, pickrng));
            { int u = max(1, maxN/2 + 3); vector<int> cnt(u,1); int rem = maxN - u; for(int i=0;i<min(rem,10); ++i) cnt[i]++; vector<int> ids(u); iota(ids.begin(), ids.end(), 1); shuffle_with_rng(ids, pickrng); reps.push_back(build_from_counts(cnt, ids, 0, pickrng)); }
        } else if(s == 5){
            reps.push_back(gen_all_distinct(maxN));
            reps.push_back(gen_all_equal(maxN,1));
            reps.push_back(make_weighted_x_test(maxN, max(2, maxN/10), 2, pickrng));
            reps.push_back(make_weighted_x_test(maxN, max(5, maxN/5), 1, pickrng));
            reps.push_back(make_each_at_most_2_test(maxN, max(1, maxN/2 - 5), 2, pickrng));
        } else if(s == 6){
            reps.push_back(gen_all_distinct(maxN));
            reps.push_back(gen_all_equal(maxN,1));
            reps.push_back(make_weighted_x_test(maxN, max(10, maxN/100), 2, pickrng));
            reps.push_back(make_weighted_x_test(maxN, max(100, maxN/50), 1, pickrng));
            reps.push_back(make_each_at_most_2_test(maxN, max(1, maxN/2 + 7), 0, pickrng));
        } else if(s == 7){
            reps.push_back(gen_all_distinct(maxN));
            reps.push_back(gen_all_equal(maxN,1));
            reps.push_back(make_weighted_x_test(maxN, max(100, maxN/1000), 2, pickrng));
            reps.push_back(make_weighted_x_test(maxN, max(1000, maxN/500), 1, pickrng));
            reps.push_back(make_each_at_most_2_test(maxN, max(1, maxN/2 + 20), 0, pickrng));
        }

        // ensure exactly 5 reps
        while(reps.size() < 5) reps.push_back(make_weighted_x_test(maxN, 2 + (int)reps.size(), (int)reps.size()%3, pickrng));
        if(reps.size() > 5) reps.resize(5);

        // add reps to all with small deterministic extra shuffle
        for(size_t i=0;i<reps.size(); ++i){
            ++global_index;
            std::mt19937_64 trng(BASE_SEED + (ull)global_index * 1234577ULL + (ull)s * 131071ULL + (ull)i * 65537ULL);
            vector<int> P = reps[i];
            // small extra shuffle to avoid exact duplicates
            if(P.size() > 1 && (i % 2 == 1)) shuffle_with_rng(P, trng);
            add(s, P);
        }

        // now add exactly 1 random test for this subtask (respecting constraints)
        ++global_index;
        std::mt19937_64 rng(BASE_SEED + (ull)global_index * 7919ULL + (ull)s * 10007ULL);
        int Nrand = 1 + (int)(rng() % maxN);

        if(s == 3){
            int count1 = (int)(rng() % (Nrand + 1));
            vector<int> P(Nrand);
            for(int i=0;i<Nrand;++i) P[i] = (i < count1 ? 1 : 2);
            int layout = (int)(rng() % 3);
            if(layout == 2) shuffle_with_rng(P, rng);
            else if(layout == 1){
                vector<int> tmp; tmp.reserve(Nrand);
                int c1=count1, c2=Nrand-count1;
                while((int)tmp.size() < Nrand){
                    if(c1>0){ tmp.push_back(1); c1--; if((int)tmp.size()==Nrand) break; }
                    if(c2>0){ tmp.push_back(2); c2--; }
                }
                P.swap(tmp);
            }
            add(s, P);
        } else if(s == 4){
            int min_u = (Nrand + 1) / 2;
            int max_u = Nrand;
            int unique_u = min_u + (int)(rng() % (max_u - min_u + 1));
            int layout = (int)(rng() % 3);
            vector<int> P = make_each_at_most_2_test(Nrand, unique_u, layout, rng);
            add(s, P);
        } else {
            int K = max(1, min(Nrand, max(10, Nrand/2)));
            vector<int> P = gen_random_values(Nrand, K, rng);
            add(s, P);
        }
    } // end for s

    // final sanity checks
    for(size_t i=0;i<all.size(); ++i){
        int idx = (int)i+1;
        int sub = all[i].first;
        const vector<int>& P = all[i].second;
        int N = (int)P.size();
        for(size_t j=0;j<P.size(); ++j){
            if(P[j] < 1 || P[j] > N){
                cerr << "ERROR: test " << idx << " value out of range: P["<<j<<"]="<<P[j]<<" not in 1.."<<N<<"\n";
                return 1;
            }
        }
        if(sub == 3){
            for(int v: P) if(!(v==1 || v==2)){
                cerr << "ERROR: test " << idx << " violates subtask3: value not in {1,2}\n"; return 1;
            }
        }
        if(sub == 4){
            unordered_map<int,int> cnt;
            for(int v: P){ cnt[v]++; if(cnt[v] > 2){ cerr << "ERROR: test " << idx << " violates subtask4: value occurs >2\n"; return 1; } }
        }
    }

    // write tests sequentially elections.1.in ... elections.M.in (shuffled per-file)
    for(size_t i=0;i<all.size(); ++i){
        write_test_file_shuffled((int)i+1, all[i].first, all[i].second);
    }

    // grade.properties (simple grouping; adapt if you need different groups)
    string groups = "1-2,3-8,9-14,15-20,21-26,27-32,33-38"; // adjust counts per subtasks if needed
    string weights = "0,10,15,15,25,25,10";
    string deps = ",1,, ,1-2,1-2,1-5";
    FILE *gf = fopen("grade.properties","w");
    if(!gf){ cerr<<"Cannot write grade.properties\n"; return 1; }
    fprintf(gf, "time=4\nmemory=1024\ngroups=%s\nweights=%s\ndependencies=%s\n", groups.c_str(), weights.c_str(), deps.c_str());
    fclose(gf);

    cerr << "Generated " << all.size() << " tests and grade.properties (subtask2: all-equal restored, permutation replaced by alternating pattern)\n";
    return 0;
}
