// gen_elections_full_fixed.cpp
// Full test generator for "elections" integrated with:
//  - log_5 generator for number of values that occur exactly 2 times (subtask 4)
//  - per-test seeds to avoid many equal tests
//  - preserved sample tests (subtask 1)
//  - special-case handling for subtask 3 (only 1/2) and subtask 4 (<=2 occurrences)
// - Fixed: random tests per subtask respect subtask-specific constraints
//
// Compile: g++ -std=c++17 -O2 gen_elections_full_fixed.cpp -o gen_elections_full_fixed
// Run: ./gen_elections_full_fixed
#include <bits/stdc++.h>
using namespace std;
using ull = unsigned long long;
using ll = long long;

const ull BASE_SEED = 123456789ULL;
static std::mt19937_64 global_rng(BASE_SEED);

const int MOD = 1'000'000'007;

// Subtask parameters (max N)
const int MAXN_SUB[8] = {0, 8, 20, 100000, 100000, 1000, 100000, 2000000};
// number of tests per subtask (default ~12). You can tune these counts.
const int TESTS_PER_SUB[8] = {0,2,12,12,12,12,12,12};

// utility rng helpers
int randint_mt(std::mt19937_64 &rng, int a, int b){
    std::uniform_int_distribution<int> d(a,b);
    return d(rng);
}
double next01(std::mt19937_64 &rng){
    std::uniform_real_distribution<double> d(nextafter(0.0,1.0), 1.0);
    return d(rng);
}
void shuffle_with_rng(vector<int>& v, std::mt19937_64 &rng){ shuffle(v.begin(), v.end(), rng); }

void write_test_file(int idx, const vector<int>& P){
    string fn = "elections." + to_string(idx) + ".in";
    FILE *f = fopen(fn.c_str(),"w");
    if(!f){ cerr<<"Cannot open "<<fn<<"\n"; exit(1); }
    fprintf(f, "%d\n", (int)P.size());
    for(size_t i=0;i<P.size();++i){
        if(i) fprintf(f, " ");
        fprintf(f, "%d", P[i]);
    }
    fprintf(f, "\n");
    fclose(f);
}

// basic building blocks (ids always in 1..N)
vector<int> gen_all_distinct(int N){
    vector<int> P(N);
    for(int i=0;i<N;++i) P[i] = i+1;
    return P;
}
vector<int> gen_all_equal(int N, int x){
    if(x < 1) x = 1;
    if(x > N) x = 1;
    return vector<int>(N, x);
}
vector<int> gen_random_values(int N, int K, std::mt19937_64 &rng){
    if(K < 1) K = 1;
    if(K > N) K = N;
    vector<int> P(N);
    for(int i=0;i<N;++i) P[i] = randint_mt(rng, 1, K);
    return P;
}

// build array from counts and ids (ids assumed 1..u)
vector<int> build_from_counts(const vector<int>& cnt, const vector<int>& ids, int layout, std::mt19937_64 &rng){
    int x = (int)cnt.size();
    int N = 0; for(int v:cnt) N += v;
    vector<int> A; A.reserve(N);
    if(layout == 0){ // block
        for(int i=0;i<x;++i) for(int k=0;k<cnt[i];++k) A.push_back(ids[i]);
    } else if(layout == 1) { // interleave round-robin
        vector<int> cur(cnt.begin(), cnt.end()); int rem = N;
        while(rem > 0){
            for(int i=0;i<x && rem>0;++i){
                if(cur[i] > 0){ A.push_back(ids[i]); cur[i]--; rem--; }
            }
        }
    } else { // shuffle
        for(int i=0;i<x;++i) for(int k=0;k<cnt[i];++k) A.push_back(ids[i]);
        shuffle_with_rng(A, rng);
    }
    return A;
}

// helper: weighted counts via Exp(1)
vector<int> gen_weighted_counts(int N, int x, bool ensure_nonzero, std::mt19937_64 &rng){
    if(x <= 0) x = 1;
    if(x > N) x = N;
    vector<double> w(x), frac(x);
    double S = 0.0;
    for(int i=0;i<x;++i){ double u = next01(rng); w[i] = -log(u); S += w[i]; }
    vector<int> cnt(x,0);
    int total = 0;
    for(int i=0;i<x;++i){
        double real = (w[i]/S) * (double)N;
        cnt[i] = (int)floor(real);
        frac[i] = real - cnt[i];
        total += cnt[i];
    }
    int rem = N - total;
    vector<int> idx(x); iota(idx.begin(), idx.end(), 0);
    sort(idx.begin(), idx.end(), [&](int a,int b){ return frac[a] > frac[b]; });
    for(int k=0;k<rem; ++k) cnt[idx[k % x]]++;
    if(ensure_nonzero && x <= N){
        vector<int> zeros;
        for(int i=0;i<x;++i) if(cnt[i] == 0) zeros.push_back(i);
        if(!zeros.empty()){
            vector<int> donors(x); iota(donors.begin(), donors.end(), 0);
            sort(donors.begin(), donors.end(), [&](int a,int b){ if(cnt[a]!=cnt[b]) return cnt[a]>cnt[b]; return a<b; });
            int di = 0;
            for(int z: zeros){
                while(di < (int)donors.size() && donors[di] == z) di++;
                while(di < (int)donors.size() && cnt[donors[di]] <= 1) di++;
                if(di >= (int)donors.size()) break;
                cnt[donors[di]]--; cnt[z]++; 
            }
            for(int z: zeros) if(cnt[z] == 0){
                int best = max_element(cnt.begin(), cnt.end()) - cnt.begin();
                if(cnt[best] > 0){ cnt[best]--; cnt[z]++; }
            }
        }
    }
    int s = accumulate(cnt.begin(), cnt.end(), 0);
    if(s != N) cnt[0] += (N - s);
    return cnt;
}
vector<int> make_weighted_x_test(int N, int x, int layout, std::mt19937_64 &rng){
    if(x <= 0) x = 1;
    if(x > N) x = N;
    vector<int> cnt = gen_weighted_counts(N, x, true, rng);
    vector<int> ids((int)cnt.size());
    for(int i=0;i<(int)ids.size();++i) ids[i] = i+1;
    return build_from_counts(cnt, ids, layout, rng);
}

// make each value occurs <= 2 test; unique_u is number of distinct ids
vector<int> make_each_at_most_2_test(int N, int unique_u, int layout, std::mt19937_64 &rng){
    unique_u = max(1, min(unique_u, N));
    vector<int> cnt(unique_u, 1);
    int rem = N - unique_u;
    vector<int> idx(unique_u); iota(idx.begin(), idx.end(), 0);
    shuffle_with_rng(idx, rng);
    for(int i=0;i<rem;++i) cnt[idx[i % unique_u]]++;
    vector<int> ids(unique_u);
    for(int i=0;i<unique_u;++i) ids[i] = i+1;
    return build_from_counts(cnt, ids, layout, rng);
}

// --- NEW: log5-x2 generator (for subtask 4) ---
// For given N and chosen x2 (# ids that occur exactly 2 times), build P:
//  - twos = x2; ones = N - 2*twos; u = twos + ones
// layout: 0 block,1 interleave,2 shuffle
vector<int> build_for_x2(int N, int x2, int layout, std::mt19937_64 &rng){
    if(x2 < 0) x2 = 0;
    int twos = min(x2, N/2);
    int ones = N - 2*twos;
    int u = twos + ones;
    if(u <= 0) u = 1;
    vector<int> cnt(u,0);
    for(int i=0;i<twos;++i) cnt[i]=2;
    for(int i=twos;i<u;++i) cnt[i]=1;
    vector<int> ids(u); iota(ids.begin(), ids.end(), 1);
    // permute ids mapping to add diversity
    shuffle_with_rng(ids, rng);
    return build_from_counts(cnt, ids, layout, rng);
}

// helper to compute pow5 list
vector<long long> pow5_list(long long max_x2){
    vector<long long> p; long long cur = 1;
    while(cur <= max_x2){ p.push_back(cur); if(cur > max_x2 / 5) break; cur *= 5; }
    return p;
}

int main(){
    ios::sync_with_stdio(false);
    cin.tie(nullptr);

    vector<pair<int, vector<int>>> all; // pair(subtask, P)
    auto add = [&](int sub, const vector<int>& P){ all.emplace_back(sub, P); };

    int global_index = 0;

    for(int s = 1; s <= 7; ++s){
        int maxN = MAXN_SUB[s];
        int total = TESTS_PER_SUB[s];

        if(s == 1){
            // exact samples from statement
            add(s, vector<int>{2,1,2}); // sample 1
            add(s, vector<int>{1,3,4,3,3,2,3,1}); // sample 2
            continue;
        }

        int nonrandom = max(0, total - 2);

        // create per-subtask RNG base to vary outputs but keep reproducible
        for(int tcase = 0; tcase < nonrandom; ++tcase){
            ++global_index;
            // per-test rng seed to avoid many identical tests across subtask
            std::mt19937_64 rng(BASE_SEED + (ull)global_index * 7919ULL + (ull)s * 10007ULL);

            if(s == 3){
                // P_i in {1,2} only (special)
                // canonical cases and log5-like variations:
                if(tcase == 0){ add(s, gen_all_equal(maxN,1)); continue; }
                if(tcase == 1){ add(s, gen_all_equal(maxN,2)); continue; }
                if(tcase == 2){ // half ones / half twos
                    vector<int> P(maxN);
                    for(int i=0;i<maxN;++i) P[i] = (i < maxN/2 ? 1 : 2);
                    shuffle_with_rng(P, rng);
                    add(s, P);
                    continue;
                }
                if(tcase == 3){ // minimal majority
                    vector<int> P(maxN);
                    for(int i=0;i<maxN;++i) P[i] = (i < maxN/2 + 1 ? 1 : 2);
                    shuffle_with_rng(P, rng);
                    add(s, P);
                    continue;
                }
                // log_5 count1 tests (count of ones chosen from powers of 5 intervals)
                int max_x = maxN;
                vector<long long> p5 = pow5_list(maxN);
                if(p5.empty()){
                    // fallback random partition
                    int count1 = 1 + (int)(rng() % maxN);
                    vector<int> P(maxN);
                    for(int i=0;i<maxN;++i) P[i] = (i < count1 ? 1 : 2);
                    shuffle_with_rng(P, rng);
                    add(s,P);
                    continue;
                }
                int idxb = (tcase - 4) % (int)p5.size();
                if(idxb < 0) idxb = 0;
                long long lo = p5[idxb];
                long long hi = min((long long)maxN, lo*5 - 1);
                if(lo > hi) lo = hi;
                long long pick = lo + (rng() % (hi - lo + 1));
                int count1 = (int)min<long long>(maxN, pick);
                vector<int> P(maxN);
                for(int i=0;i<maxN;++i) P[i] = (i < count1 ? 1 : 2);
                // vary layout
                int layout = (tcase % 3);
                if(layout == 2) shuffle_with_rng(P, rng);
                else if(layout == 1){
                    vector<int> tmp; tmp.reserve(maxN);
                    int c1=count1, c2=maxN-count1;
                    while((int)tmp.size() < maxN){
                        if(c1>0){ tmp.push_back(1); c1--; if((int)tmp.size()==maxN) break; }
                        if(c2>0){ tmp.push_back(2); c2--; }
                    }
                    P.swap(tmp);
                }
                add(s,P);
                continue;
            } // s==3

            if(s == 4){
                // subtask 4: each value occurs <= 2
                // integrate log5-x2 here plus some offsets for diversity
                int which = tcase;
                if(which == 0){
                    // all distinct
                    add(s, gen_all_distinct(maxN));
                    continue;
                }
                if(which == 1){
                    // exact each value twice: x2 = N/2
                    int x2 = maxN/2;
                    vector<int> P = build_for_x2(maxN, x2, 2, rng);
                    add(s, P);
                    continue;
                }

                // create pow5 list up to floor(N/2)
                int max_x2 = maxN / 2;
                vector<long long> p5 = pow5_list(max_x2);
                if(!p5.empty()){
                    int idxb = (which - 2) % (int)p5.size();
                    if(idxb < 0) idxb = 0;
                    long long lo = p5[idxb];
                    long long hi = min((long long)max_x2, lo * 5 - 1);
                    if(lo > hi) lo = hi;
                    long long pick;
                    if(((which - 2) % 3) == 0) pick = (lo + hi) / 2;
                    else pick = lo + (rng() % (hi - lo + 1));
                    int x2 = (int)min<long long>(max_x2, pick);
                    int layout = (which-2) % 3;
                    vector<int> P = build_for_x2(maxN, x2, layout, rng);
                    add(s, P);
                    continue;
                } else {
                    // fallback: produce some each_at_most_2 test with varying u
                    int u = max(1, maxN/2 + ((which-2)%11) - 5);
                    vector<int> P = make_each_at_most_2_test(maxN, u, (which%3), rng);
                    add(s, P);
                    continue;
                }
            } // s==4

            // default path for other subtasks (2,5,6,7)
            if(s == 2){
                // small N (<=20), create varied set to avoid many equal answers:
                if(tcase == 0){ add(s, gen_all_distinct(maxN)); continue; }
                if(tcase == 1){ add(s, gen_all_equal(maxN,1)); continue; }
                // log2 series: choose x in [2^b, 2^{b+1}-1]
                int base = 2;
                int Bmax = 0;
                { int p=1; while(p*base <= maxN){ p*=base; Bmax++; } }
                int b = (tcase - 2) % max(1, Bmax);
                if(b < 0) b = 0;
                int lo = 1<<b;
                int hi = min(maxN, (1<<(b+1)) - 1);
                int x = lo + (rng() % (hi - lo + 1));
                vector<int> P = make_weighted_x_test(maxN, x, (tcase%3), rng);
                add(s, P);
                continue;
            }

            // s == 5 or 6 or 7 default templates
            if(tcase == 0){ add(s, gen_all_distinct(maxN)); continue; }
            if(tcase == 1){ add(s, gen_all_equal(maxN, 1)); continue; }
            // generic log-base weighted tests
            int base = 5;
            // pick b according to tcase
            long long p = 1;
            int b = 0;
            while(b < 40 && p*base <= maxN && (b < (tcase-2))) { p *= base; b++; }
            long long lo = p;
            long long hi = min((long long)maxN, p*(long long)base - 1);
            if(lo < 1) lo = 1;
            if(hi < lo) hi = lo;
            long long pick = lo + (rng() % (hi - lo + 1));
            int x = (int)min<long long>(maxN, pick);
            vector<int> P = make_weighted_x_test(maxN, x, (tcase%3), rng);
            add(s, P);
        }

        // finally add two random tests per subtask (random Nrand) — FIXED: respect subtask constraints
        for(int r=0; r<2; ++r){
            ++global_index;
            std::mt19937_64 rng(BASE_SEED + (ull)global_index * 7919ULL + (ull)s * 10007ULL + 1234ULL);
            int Nrand = 1 + (int)(rng() % maxN);

            if(s == 3){
                // subtask 3: values must be only 1 or 2
                int count1 = (int)(rng() % (Nrand + 1)); // #ones in [0..Nrand]
                int layout = (int)(rng() % 3);
                vector<int> P(Nrand);
                for(int i=0;i<Nrand;++i) P[i] = (i < count1 ? 1 : 2);
                if(layout == 2) shuffle_with_rng(P, rng);
                else if(layout == 1){
                    vector<int> tmp; tmp.reserve(Nrand);
                    int c1 = count1, c2 = Nrand - count1;
                    while((int)tmp.size() < Nrand){
                        if(c1>0){ tmp.push_back(1); c1--; if((int)tmp.size()==Nrand) break; }
                        if(c2>0){ tmp.push_back(2); c2--; }
                    }
                    P.swap(tmp);
                }
                add(s, P);
            } else if(s == 4){
                // subtask 4: each value occurs at most 2 times
                int min_u = (Nrand + 1) / 2;
                int max_u = Nrand;
                int unique_u = min_u + (int)(rng() % (max_u - min_u + 1));
                int layout = (int)(rng() % 3);
                vector<int> P = make_each_at_most_2_test(Nrand, unique_u, layout, rng);
                add(s, P);
            } else {
                // general case: arbitrary values in 1..K
                int K = max(1, min(Nrand, max(10, Nrand/2)));
                vector<int> P = gen_random_values(Nrand, K, rng);
                add(s, P);
            }
        }
    } // end for s

    // sanity checks: ensure subtask-specific constraints
    for(size_t i=0;i<all.size();++i){
        int idx = (int)i+1;
        int sub = all[i].first;
        const vector<int>& P = all[i].second;
        int N = (int)P.size();
        // bounds
        for(size_t j=0;j<P.size();++j){
            if(P[j] < 1 || P[j] > N){
                cerr << "ERROR: test " << idx << " value out of range: P["<<j<<"]="<<P[j]<<" not in 1.."<<N<<"\n";
                return 1;
            }
        }
        if(sub == 3){
            for(int v: P) if(!(v==1 || v==2)){
                cerr << "ERROR: test " << idx << " violates subtask3: value not in {1,2}\n"; return 1;
            }
        }
        if(sub == 4){
            unordered_map<int,int> cnt;
            for(int v: P){ cnt[v]++; if(cnt[v] > 2){ cerr << "ERROR: test " << idx << " violates subtask4: value occurs >2\n"; return 1; } }
        }
    }

    // write files grouped by subtask order (already in order we added them)
    for(size_t i=0;i<all.size(); ++i){
        write_test_file((int)i+1, all[i].second);
    }

    // grade.properties (groups must align with number of tests)
    // Groups chosen to match our count (2 + 12*6 = 74 typical)
    string groups = "1-2,3-14,15-26,27-38,39-50,51-62,63-74";
    string weights = "0,10,15,15,25,25,10";
    string dependencies = ",1,, ,1-2,1-2,1-5";
    // clean dependencies (remove spaces)
    string deps; for(char c: dependencies) if(c!=' ') deps.push_back(c);

    FILE *gf = fopen("grade.properties","w");
    if(!gf){ cerr<<"Cannot write grade.properties\n"; return 1; }
    fprintf(gf, "time=4\n");
    fprintf(gf, "memory=1024\n");
    fprintf(gf, "groups=%s\n", groups.c_str());
    fprintf(gf, "weights=%s\n", weights.c_str());
    fprintf(gf, "dependencies=%s\n", deps.c_str());
    fclose(gf);

    cerr << "Generated " << all.size() << " tests (elections.1.in .. elections." << all.size() << ".in) and grade.properties\n";
    return 0;
}
